<?php

require_once 'Teamnet/Fax/Soap/Type/QuickstartTypes.php';

/**
 * Client class for QuickStart related tasks, using the Teamnet FaxAPI webservice
 * @copyright Teamnet GmbH
 * @version 1.1
 */
class Teamnet_Fax_Soap_Client_Quickstart {
	/**
	 * @var SoapClient
	 */
	public $oSOAP;
	
	
	
	/**
	 * @throws Teamnet_Fax_Soap_Type_ServiceException
	 * @throws ErrorException
	 */
	public function __construct($wsdl ) {
		try {
			if ( !isset( $wsdl ) ) {
				throw new ErrorException( 'WSDL isn\'t set.' );
			}
			$this->oSOAP = new SoapClient( (string) $wsdl, array( 'classmap' => getFaxapiQuickstartClassMap() ));
		}
		catch ( SoapFault $exception ) {
			throw new Teamnet_Fax_Soap_Type_ServiceException( $exception );
		}
	}
	
	
	
	/**
	 * Checks whether the given AuthKey is valid or not
	 * @param string $authKey
	 * @return bool
	 * @throws Teamnet_Fax_Soap_Type_ServiceException		if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException	if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException		if parameters are invalid
	 * @throws SoapFault									if another SOAP related error occurred
	 */
	public function checkAuthentification($authKey) {
		$oCheckAuthRequest = new Teamnet_Fax_Soap_Type_CheckAuthentificationRequest();
		$oCheckAuthRequest->authKey = $authKey;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->oSOAP->checkAuthentification($oCheckAuthRequest);
		} catch (SoapFault $e) {
			if (! isset($e->detail)) {
				throw $e;
			}
			
			$oException = $this->_convertSoapFault($e);
			
			if ($oException instanceof Teamnet_Fax_Soap_Type_AuthentificationException) {
				return false;
			}
			
			throw $oException;
		}
		
		return true;
	}
	
	
	
	/**
	 * Used by QuickStart to check for a new version
	 * @param string		$sVersion
	 * @param string|null	$sSysInfo
	 * @throws Teamnet_Fax_Soap_Type_ServiceException		if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException	if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException		if parameters are invalid
	 * @throws SoapFault									if another SOAP related error occurred
	 */
	public function checkVersion($sVersion, $sSysInfo = null) {
		$req = new Teamnet_Fax_Soap_Type_CheckVersionRequest();
		$req->system = 'PHP';
		$req->qsVersion = $sVersion;
		$req->systemInfo = $sSysInfo;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			return $this->oSOAP->checkVersion($req);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
	}
	
	
	
	/**
	 * @param SoapFault $oSoapFault
	 * @return mixed
	 * @internal
	 */
	private function _convertSoapFault(SoapFault $oSoapFault) {
		$m_Detail = $oSoapFault->detail;
		$a_Detail = (array)$m_Detail;
		
		$keys = array_keys($a_Detail);
		$s_ExceptionType = 'Teamnet_Fax_Soap_Type_'.array_pop($keys );
		
		$o_Exception = new $s_ExceptionType();
		$o_Exception->parseSOAPFault($oSoapFault);
		
		return $o_Exception;
	}
}
