<?php

require_once 'Teamnet/Fax/Soap/Type/JobTypes.php';

/**
 * Client class to obtain or alter faxjobs using the Teamnet FaxAPI webservice
 * @copyright Teamnet GmbH
 * @version 1.1
 */
class Teamnet_Fax_Soap_Client_Job {
	/**
	 * @var SoapClient
	 */
	private $oSOAP;
	private $authKey;
	
	
	/**
	 * @throws Teamnet_Fax_Soap_Type_ServiceException
	 * @throws ErrorException
	 */
	public function __construct($wsdl, $authKey ) {
		try {
			if ( !isset( $wsdl ) ) {
				throw new ErrorException( 'WSDL isn\'t set.' );
			}
			$this->authKey = $authKey;
			$this->oSOAP = new SoapClient( (string) $wsdl, array( 'classmap' => getFaxapiJobClassMap() ) );
		}
		catch ( SoapFault $exception ) {
			throw new Teamnet_Fax_Soap_Type_ServiceException( $exception );
		}
	}
	
	
	
	/**
	 * Create a period object
	 *  
	 * @param string $sFrom Begin of the period formatted as ISO 8601 date
	 * @param string $sUntil End of the period formatted as ISO 8601 date
	 */
	public static function createPeriod( $sFrom, $sUntil ) {
		$oPeriod = new Teamnet_Fax_Soap_Type_Period();
		$oPeriod->from = $sFrom;
		$oPeriod->until = $sUntil;
		
		return $oPeriod;
	}
	
	
	
	/**
	 * Gets a list of members associated to a specific job
	 * @param int $iJobID
	 * @return array of APIJobMemberList objects
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_JobNotFoundException		if a job with the given ID was not found
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function getJobMembers($iJobID) {
		$oReq = new Teamnet_Fax_Soap_Type_GetJobMembers();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_GetJobMembersResponse $oResponse */
			$oResponse = $this->oSOAP->getJobMembers($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->apiJobMemberList->apiJobMemberList;
	}
	
	
	
	/**
	 * Gets the costs of a specific job
	 * @param int $iJobID
	 * @return float|null
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_JobNotFoundException		if a job with the given ID was not found
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function getJobCosts($iJobID) {
		$oReq = new Teamnet_Fax_Soap_Type_GetJobCosts();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_GetJobCostsResponse $oResponse */
			$oResponse = $this->oSOAP->getJobCosts($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->costs;
	}
	
	
	
	/**
	 * Gets the current status of a specific job
	 * @param int $iJobID JobID
	 * @return Teamnet_Fax_Soap_Type_APIJob
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_JobNotFoundException		if a job with the given ID was not found
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function getJob($iJobID) {
		$oReq = new Teamnet_Fax_Soap_Type_GetJob();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_GetJobResponse $oResponse */
			$oResponse = $this->oSOAP->getJob($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->apiJob;
	}
	
	
	
	/**
	 *
	 * @param int $iJobID JobID
	 * @return Teamnet_Fax_Soap_Type_APIJobProfile
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_JobNotFoundException		if a job with the given ID was not found
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function getJobProfile($iJobID) {
		$oReq = new Teamnet_Fax_Soap_Type_GetJobProfile();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_GetJobProfileResponse $oResponse */
			$oResponse = $this->oSOAP->getJobProfile($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->apiJobProfile;
	}
	
	
	
	/**
	 * Returns a list of APIJob objects matching the delivered search parameters
	 * @param string|null	$sTitle
	 * @param string|null	$sStatus
	 * @param Teamnet_Fax_Soap_Type_Period|null	$oPeriod
	 * @return array						APIJob objects
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function getList($sTitle = null, $sStatus = null, Teamnet_Fax_Soap_Type_Period $oPeriod = null) {
		$oReq = new Teamnet_Fax_Soap_Type_GetList();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sTitle;
		$oReq->status = $sStatus;
		$oReq->period = $oPeriod;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_JobList $jobList */
			$jobList = $this->oSOAP->getList($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $jobList->jobList->jobList;
	}
	
	
	
	/**
	 * Returns a list of APIJobMember objects matching the delivered search parameters
	 * @param string|null	$sTitle
	 * @param string|null	$sStatus
	 * @param Teamnet_Fax_Soap_Type_Period|null	$oPeriod
	 * @return array						APIJobMember objects
	 * @throws Teamnet_Fax_Soap_Type_ServiceException				if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException			if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException				if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException		if the given AuthKey is invalid
	 * @throws SoapFault											if another SOAP related error occurred
	 */
	public function getMembers($sTitle = null, $sStatus = null, Teamnet_Fax_Soap_Type_Period $oPeriod = null) {
		$oReq = new Teamnet_Fax_Soap_Type_GetMembers();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sTitle;
		$oReq->status = $sStatus;
		$oReq->period = $oPeriod;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_GetMembersResponse $jobList */
			$jobList = $this->oSOAP->getMembers($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $jobList->apiJobMemberList->apiJobMemberList;
	}
	
	
	
	/**
	 * Gets the current transport state of a specific job
	 * @param int $iJobID
	 * @return string	the current transport status
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_JobNotFoundException		if a job with the given ID was not found
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function getTransportStatus($iJobID) {
		$oReq = new Teamnet_Fax_Soap_Type_GetTransportStatus();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_GetTransportStatusResponse $oResponse */
			$oResponse = $this->oSOAP->getTransportStatus($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->transportStatus;
	}
	
	
	
	/**
	 * Alters the transport status of a specific job
	 * @param int		$iJobID
	 * @param string	$sTransportStatus	the transport status to set
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_JobNotFoundException		if a job with the given ID was not found
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function setTransportStatus($iJobID, $sTransportStatus) {
		$oReq = new Teamnet_Fax_Soap_Type_SetTransportStatus();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		$oReq->transportStatus = $sTransportStatus;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->oSOAP->setTransportStatus($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
	}
	
	
	
	/**
	 * @param SoapFault $oSoapFault
	 * @return mixed
	 * @internal
	 */
	private function _convertSoapFault(SoapFault $oSoapFault) {
		$m_Detail = $oSoapFault->detail;
		$a_Detail = (array)$m_Detail;
		
		$keys = array_keys($a_Detail);
		$s_ExceptionType = 'Teamnet_Fax_Soap_Type_'.array_pop($keys );
		
		$o_Exception = new $s_ExceptionType();
		$o_Exception->parseSOAPFault($oSoapFault);
		
		return $o_Exception;
	}
}
