<?php

require_once 'Teamnet/Fax/Soap/Type/SendfaxTypes.php';
require_once 'Teamnet/Fax/Soap/Client/RecipientList.php';

/**
 * Client class to send faxes using the Teamnet FaxAPI webservice
 * @copyright Teamnet GmbH
 * @version 1.1
 */
class Teamnet_Fax_Soap_Client_SendFax {
	/**
	 * @var SoapClient
	 */
	private $oSOAP;
	private $authKey;
	
	
	
	/**
	 * @throws Teamnet_Fax_Soap_Type_ServiceException
	 * @throws ErrorException
	 */
	public function __construct($wsdl, $authKey ) {
		try {
			if ( !isset( $wsdl ) ) {
				throw new ErrorException( 'WSDL isn\'t set.' );
			}
			
			$this->authKey = $authKey;
			$this->oSOAP = new SoapClient( (string) $wsdl, array( 'classmap' => getFaxapiSendfaxClassMap() ) );
		} catch ( SoapFault $exception ) {
			throw new Teamnet_Fax_Soap_Type_ServiceException( $exception );
		}
	}
	
	
	
	/**
	 * Sends a fax to one specified recipient
	 * @param string			$sJobTitle
	 * @param string			$sContentType
	 * @param string			$sContent
	 * @param string			$sFaxnumber		Faxnummer
	 * @param Teamnet_Fax_Soap_Type_JobOptions|null	$oJobOptions
	 * @return int JobID
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_LimitExceededException		if you try to send to many jobs in a short time
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function sendFaxToFaxnumber($sJobTitle, $sContentType, $sContent,
									   $sFaxnumber, Teamnet_Fax_Soap_Type_JobOptions $oJobOptions = null)
	{
		$oReq = new Teamnet_Fax_Soap_Type_SendFaxToFaxnumber();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sJobTitle;
		$oReq->contentType = $sContentType;
		$oReq->content = $sContent;
		$oReq->faxnumber = $sFaxnumber;
		$oReq->jobOptions = $oJobOptions;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_SendFaxResponse $oResponse */
			$oResponse = $this->oSOAP->sendFaxToFaxnumber($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->jobId;
	}
	
	
	
	/**
	 * Sends a fax to a list of recipients
	 * @param string								$sJobTitle
	 * @param string|int							$mContentType
	 * @param string								$sContent
	 * @param Teamnet_Fax_Soap_Client_RecipientList	$oRecipientList	Array von FaxRecipient objekten
	 * @param Teamnet_Fax_Soap_Type_JobOptions		$oJobOptions
	 * @return int JobID
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_LimitExceededException		if you try to send to many jobs in a short time
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function sendFaxToRecipientList($sJobTitle, $mContentType, $sContent,
										   Teamnet_Fax_Soap_Client_RecipientList $oRecipientList,
										   Teamnet_Fax_Soap_Type_JobOptions $oJobOptions = null)
	{
		$oReq = new Teamnet_Fax_Soap_Type_SendFaxToFaxRecipientList();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sJobTitle;
		$oReq->contentType = $mContentType;
		$oReq->content = $sContent;
		$oReq->faxRecipientList = $oList = new Teamnet_Fax_Soap_Type_FaxRecipientList();
		$oReq->jobOptions = $oJobOptions;
		
		$oList->faxRecipient = $oRecipientList->getRecipients();
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_SendFaxResponse $oResponse */
			$oResponse = $this->oSOAP->sendFaxToFaxRecipientList($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->jobId;
	}
	
	
	
	/**
	 * Sends a fax to a distribution list
	 * @param string			$sJobTitle
	 * @param string|int		$mContentType
	 * @param string			$sContent
	 * @param int				$iDistributionListId	ID of distribution list
	 * @param Teamnet_Fax_Soap_Type_JobOptions|null	$oJobOptions
	 * @return int JobID
	 * @throws Teamnet_Fax_Soap_Type_ServiceException			if an internal error in the fax service occurred
	 * @throws Teamnet_Fax_Soap_Type_MaintenanceException		if the fax service is currently down for maintenance
	 * @throws Teamnet_Fax_Soap_Type_ParameterException			if parameters are invalid
	 * @throws Teamnet_Fax_Soap_Type_AuthentificationException	if the given AuthKey is invalid
	 * @throws Teamnet_Fax_Soap_Type_LimitExceededException		if you try to send to many jobs in a short time
	 * @throws SoapFault										if another SOAP related error occurred
	 */
	public function sendFaxToDistributionList($sJobTitle, $mContentType, $sContent,
											  $iDistributionListId, Teamnet_Fax_Soap_Type_JobOptions $oJobOptions = null)
	{
		$oReq = new Teamnet_Fax_Soap_Type_SendFaxToDistributionList();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sJobTitle;
		$oReq->contentType = $mContentType;
		$oReq->content = $sContent;
		$oReq->distributionListId = $iDistributionListId;
		$oReq->jobOptions = $oJobOptions;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var Teamnet_Fax_Soap_Type_SendFaxResponse $oResponse */
			$oResponse = $this->oSOAP->sendFaxToDistributionList($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->jobId;
	}
	
	
	
	/**
	 * @internal
	 * @param SoapFault $oSoapFault
	 */
	private function _convertSoapFault(SoapFault $oSoapFault) {
		$m_Detail = $oSoapFault->detail;
		$a_Detail = (array)$m_Detail;
		
		$keys = array_keys($a_Detail);
		$s_ExceptionType = 'Teamnet_Fax_Soap_Type_'.array_pop($keys );
		
		$o_Exception = new $s_ExceptionType();
		$o_Exception->parseSOAPFault($oSoapFault);
		
		return $o_Exception;
	}
}
